#ifndef _NEW_DB_CPP
#define _NEW_DB_CPP
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <Windows.H>
#include <WindowsX.H>
#include <ShellAPI.H>
#include <Stdio.H>
#include <Stdlib.H>
#include <SQL.H>
#include <SQLExt.H>

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include "../Resources/Resource.H"

#include "../../SharedClasses/CRC32/CRC.H"
#include "../../SharedClasses/SQLClass/cSQL.H"
#include "../../SharedClasses/SQLClass/cRecordSet.H"

#include "../CSockSrvr/CSockSrvr.H"

#include "../Dialogs/CStatusDlg.H"

#include "NSWFL.H"
#include "Init.H"
#include "Entry.H"
#include "Routines.H"
#include "Command.H"
#include "Console.H"
#include "SQLImport.H"
#include "NewDB.H"

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool SetupNewClient(char *sClientDBFileName, char *sDBO, char *sDBName, bool bImportStructure, 
					bool bImportData, bool bDropImportTable, CStatusDlg *lpDlg)
{
	/*
		sClientDBFileName: Contains the name of the backup file that contains the users existing
			data and structure. (Relative to the server)

		sDBName: Contains the base name of the new DB to create for the new client.

		bImportStructure: Should we import the structure of the sClientDBFileName to our new DB?

		bImportData: Should we import the data from the sClientDBFileName to our new DB?
	*/

    CSQL MySQL;

	char sSQL[1024];

	if(bImportData && !bImportStructure)
	{
		lpDlg->SetText("You cannot import data if you opt not to inport the DB structure.");
		return false;
	}

	lpDlg->SetText("Connecting to the SQL server.");
	if(!MySQL.Connect(gsSQLCustDriver, gsSQLCustServer, gsSQLCustUserID, gsSQLCustPassword, "Master"))
    {
        lpDlg->SetText("Error connecting to the [Master] database.");
        return false;
    }

	if(bImportStructure || bImportData)
	{
		lpDlg->SetText("Restoring the import database. (This could take some time)");
		if(!RestoreDBForImport(&MySQL, sClientDBFileName, sDBName))
		{
			lpDlg->SetText("Failed to restore the clients database.");
			MySQL.Disconnect();
			return false;
		}
	}

	lpDlg->SetText("Creating the new client database.");
	if(!CreateNewDB(&MySQL, sDBName))
	{
		lpDlg->SetText("Failed to create the new database.");
		MySQL.Disconnect();
		return false;
	}


	if(bImportStructure || bImportData)
	{
		lpDlg->SetText("Importing the database structure.");
		if(!ImportDBStructure(&MySQL, sDBO, sDBName, bImportData, lpDlg))
		{
			lpDlg->SetText("Failed to import the database structure.");
			MySQL.Disconnect();
			return false;
		}
	}

	if(bDropImportTable)
	{
		lpDlg->SetText("Dropping the import database.");
		sprintf(sSQL, "DROP DATABASE SQLExch_%s_IMP_Temp", sDBName);
		MySQL.ExecuteNonQuery(sSQL);
	}

	MySQL.Disconnect();

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool ImportDBStructure(CSQL *lpMySQL, char *sDBO, char *sDBName, bool bImportData, CStatusDlg *lpDlg)
{
	char *sBaseSQL = NULL;
	char sSQL[5120];

	int iTempSz = 0;

	char *sLargeSQL = NULL;
	char sTable[1024];
	long lTableID = 0;

	char sColName[1024];
	char sColType[255];
	long lColLen = 0;
	long lColStatus = 0;
	long lColNullable = 0;

	char sSmlTmp[32];
	char sLgTmp[1024];

	char sPKName[255];
	char **sPKs = NULL;
	int iPKs = 0;
	int iPK = 0;
	int iRow = 0;

	CRecordSet rsTables;
	CRecordSet rsColumns;

	sBaseSQL = "SELECT Name, ID FROM SQLExch_%s_IMP_Temp.%s.SysObjects"
		" WHERE xType = 'U' AND Name <> 'dtProperties' AND LEFT(Name, 8) <> 'SQLExch_'"
		" ORDER BY Name";
	sprintf(sSQL, sBaseSQL, sDBName, gsDefaultDBO);

	if(!lpMySQL->Execute(sSQL, &rsTables))
	{
		lpMySQL->GetErrorMessage(sLgTmp, rsTables.hSTMT);
		MsgBox(sLgTmp);
		return false;
	}

	lpDlg->SetProgressRange(0, rsTables.RowCount);
	
	sLargeSQL = (char *) calloc(1024 * 1024, sizeof(char));

	while(rsTables.Fetch())
	{
		rsTables.sColumnEx(1, sTable, sizeof(sTable), &iTempSz);
		lTableID = rsTables.lColumn(2);

		sBaseSQL = "SELECT xSysColumns.Name, xSysColumns.Length, xSysTypes.Name, xSysTypes.Status, xSysColumns.IsNullable"
			" FROM SQLExch_%s_IMP_Temp.[%s].SysColumns as xSysColumns, SQLExch_%s_IMP_Temp.[%s].SysTypes as xSysTypes"
			" WHERE ID = %d AND xSysColumns.xType = xSysTypes.xUserType"
			" ORDER BY ColOrder";
		sprintf(sSQL, sBaseSQL, sDBName, sDBO, sDBName, sDBO, lTableID);

		if(!lpMySQL->Execute(sSQL, &rsColumns))
		{
			lpMySQL->GetErrorMessage(sLgTmp, rsColumns.hSTMT);
			MsgBox(sLgTmp);

			rsTables.Close();
			free(sLargeSQL);
			return false;
		}

		//------------------------------------------------------------------------------------------
		//Drop the existing table
		//------------------------------------------------------------------------------------------
		lpMySQL->bThrowErrors = false;
		sprintf(sSQL, "DROP TABLE [SQLExch_%s].[%s].[%s]", sDBName, sDBO, sTable);
		lpMySQL->ExecuteNonQuery(sSQL);
		lpMySQL->bThrowErrors = true;
		//------------------------------------------------------------------------------------------

		sprintf(sLargeSQL, "CREATE TABLE [SQLExch_%s].[%s].[%s] (\r\n", sDBName, sDBO, sTable);

		sprintf(sLgTmp, "SQLExch_%s_IMP_Temp", sDBName);
		sPKs = GetPrimaryKeysEx(lpMySQL, sTable, sDBO, sLgTmp, &iPKs, sPKName);

		while(rsColumns.Fetch())
		{
			rsColumns.sColumnEx(1, sColName, sizeof(sColName), &iTempSz);
			lColLen = rsColumns.lColumn(2);
			rsColumns.sColumnEx(3, sColType, sizeof(sColType), &iTempSz);
			lColStatus = rsColumns.lColumn(4);
			lColNullable = rsColumns.lColumn(5);

			strcat(sLargeSQL, "\t[");
			strcat(sLargeSQL, sColName);
			strcat(sLargeSQL, "] [");
			strcat(sLargeSQL, sColType);
			strcat(sLargeSQL, "]");
			if(lColStatus != 0)
			{
				sprintf(sSmlTmp, " (%d)", lColLen);
				strcat(sLargeSQL, sSmlTmp);
			}
			
			if(lColNullable == 0)
			{
				strcat(sLargeSQL, " NOT NULL,\r\n");
			}
			else {
				strcat(sLargeSQL, " NULL,\r\n");
			}
		}

		iTempSz = strlen(sLargeSQL);
		sLargeSQL[iTempSz - 1] = '\0';
		sLargeSQL[iTempSz - 2] = '\0';
		sLargeSQL[iTempSz - 3] = '\0';
		strcat(sLargeSQL, "\r\n");

		if(iPKs > 0)
		{
			strcat(sLargeSQL, "\tCONSTRAINT [");
			strcat(sLargeSQL, sPKName);
			strcat(sLargeSQL, "] PRIMARY KEY CLUSTERED\r\n");
			strcat(sLargeSQL, "\t(\r\n");

			iPK = 0;
			while(iPK < iPKs)
			{
				strcat(sLargeSQL, "\t\t[");
				strcat(sLargeSQL, sPKs[iPK]);
				strcat(sLargeSQL, "]");

				if(iPK != iPKs - 1)
				{
					strcat(sLargeSQL, ",\r\n");
				}
				iPK++;
			}
			strcat(sLargeSQL, "\r\n\t) ON [PRIMARY]\r\n");
			
		}

		strcat(sLargeSQL, ") ON [PRIMARY]");

		//MsgBox(sLargeSQL);
		lpMySQL->ExecuteNonQuery(sLargeSQL);

		if(bImportData)
		{
			//------------------------------------------------------------------------------------------
			//Import the data.
			//------------------------------------------------------------------------------------------
			sprintf(sSQL, "INSERT INTO [SQLExch_%s].[%s].[%s] SELECT * FROM [SQLExch_%s_IMP_Temp].[%s].[%s]",
				sDBName, sDBO, sTable, sDBName, sDBO, sTable);
			lpMySQL->ExecuteNonQuery(sSQL);
			//------------------------------------------------------------------------------------------
		}

		FreePrimaryKeys(sPKs, iPKs);
		rsColumns.Close();

		iRow++;

		lpDlg->SetProgressPos(iRow);
	}

	free(sLargeSQL);
	rsTables.Close();

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool RestoreDBForImport(CSQL *lpMySQL, char *sClientDBFileName, char *sDBName)
{
	char *sBaseSQL = NULL;
	char sSQL[5120];

	int iTempSz = 0;
	char sTemp1[1024];
	char sTemp2[1024];

	char sLogicalData[1024];
	char sLogicalLog[1024];

	CRecordSet rsTemp;

	sprintf(sSQL, "RESTORE FILELISTONLY FROM DISK = '%s'", sClientDBFileName);
	if(!lpMySQL->Execute(sSQL, &rsTemp))
	{
		MsgBox("The backup file specified is either invalid or does not exist.");
		return false;
	}

	strcpy(sLogicalData, "");
	strcpy(sLogicalLog, "");

	while(rsTemp.Fetch())
	{
		rsTemp.sColumnEx(1, sTemp1, sizeof(sTemp1), &iTempSz);

		rsTemp.sColumnEx(3, sTemp2, sizeof(sTemp2), &iTempSz);
		if(strcmpi(sTemp2, "D") == 0)
		{
			//Data File.
			strcpy(sLogicalData, sTemp1);
		}
		else if(strcmpi(sTemp2, "L") == 0)
		{
			//Log File.
			strcpy(sLogicalLog, sTemp1);
		}
	}

	rsTemp.Close();

	if(strlen(sLogicalData) > 0 && strlen(sLogicalLog) > 0)
	{

		sBaseSQL = "RESTORE DATABASE SQLExch_%s_IMP_Temp FROM DISK = '%s'\n"
			"WITH MOVE '%s' TO '%s\\SQLExch_%s_IMP_Temp_Dat.mdf',\n"
			"MOVE '%s' TO '%s\\SQLExch_%s_IMP_Temp_Log.ldf'\n";

		sprintf(sSQL, sBaseSQL, sDBName, sClientDBFileName,
			sLogicalData, gsSQLDataFiles, sDBName,
			sLogicalLog, gsSQLDataFiles, sDBName);
	}
	else{
		MsgBox("Failed to obtain a list of logical file names from the backup file specified.");
		return false;
	}

	return lpMySQL->ExecuteNonQuery(sSQL);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool CreateNewDB(CSQL *lpMySQL, char *sDBName)
{
	char *sBaseSQL = NULL;
	char sSQL[5120];

	sBaseSQL = "CREATE DATABASE SQLExch_%s\r\n"
		"ON \r\n"
		"( NAME = 'SQLExch_%s_Dat',\r\n"
		"	FILENAME = '%s\\SQLExch_%s_Dat.mdf',\r\n"
		"	SIZE = 10,\r\n"
		"	FILEGROWTH = 5 )\r\n"
		"LOG ON\r\n"
		"( NAME = 'SQLExch_%s_Log',\r\n"
		"	FILENAME = '%s\\SQLExch_%s_Log.ldf',\r\n"
		"	SIZE = 5MB,\r\n"
		"	FILEGROWTH = 5MB)\r\n";

	sprintf(sSQL, sBaseSQL, sDBName, sDBName, gsSQLDataFiles, sDBName, sDBName, gsSQLDataFiles, sDBName);

	return lpMySQL->ExecuteNonQuery(sSQL);
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#endif
